﻿using System;
using System.Collections;
using System.Windows.Forms;
using GTA;

namespace Airstrike {
    public class Airstrike : Script {
        const int NUM_EXPLOSIONS = 25;  // create number of explosions on the ground during airstrike

        const float STRIKE_DISTANCE      = 3.0f;   // strike when reaching this distance from the destination on the XY plane
        const float DELETE_DISTANCE      = 175.0f; // delete plane when reaching this distance from the destination on the XY plane
        const float EXPLOSION_OFFSET     = 2.5f;   // area around the destination to hit with the airstrike
        const float PLANE_START_OFFSET   = 30.0f;  // area around the destination at which the plane should be spawned
        const float PLANE_START_Z_OFFSET = 60.0f;  // distance from the ground at which the plane should be spawned
        const float DESTINATION_Z_OFFSET = 35.0f;  // distance from the ground at which the plane should strike

        ArrayList pilots;         // list of Peds in the plane including pilot and passengers
        Vehicle plane;            // the plane
        Blip planeBlip;           // the GPS blip attached to the plane
        Vector3 destination;      // the plane's destination point
        TaskSequence FlySequence; // TaskSequence for pilot to fly to destination then random point on map
        bool explosionDone;       // has the airstrike completed?

        public Airstrike() {
            Interval = 500; // run tick every 500ms

            pilots = new ArrayList();

            Settings.Load();

            bool useConsoleCommand = Settings.GetValueBool("USE_CONSOLE_COMMAND", "SETTINGS", true);
            bool useKeyCommand = Settings.GetValueBool("USE_KEY_COMMAND", "SETTINGS", false);

            if(useConsoleCommand) {
                BindConsoleCommand(Settings.GetValueString("CONSOLE_COMMAND", "SETTINGS", "airstrike"), new ConsoleCommandDelegate(SpawnPlane));
            }

            if(useKeyCommand) {
                BindKey(Settings.GetValueKey("KEY_COMMAND", "SETTINGS", Keys.F1), new KeyPressDelegate(SpawnPlane));
            }
            
            this.Tick += new EventHandler(this.Airstrike_Tick);
        }

        private void Airstrike_Tick(object sender, EventArgs e) {
            if(Exists(plane)) {
                float distance = destination.DistanceTo2D(plane.Position);
                
                if(!explosionDone && distance <= STRIKE_DISTANCE) { // if the plane is in striking distance and hasn't already strucks
                    Vector3 explosionLoc = destination;

                    for(int i = 1; i <= NUM_EXPLOSIONS; i++) { // create NUM_EXPLOSIONS number of explosions around the destination
                        explosionLoc = explosionLoc.Around(EXPLOSION_OFFSET).ToGround();
                        World.AddExplosion(explosionLoc, ExplosionType.Rocket, 500.0f);
                    }

                    explosionDone = true; // set so the plane doesn't strike again
                    GTA.Native.Function.Call("SWITCH_OFF_WAYPOINT"); // delete the waypoint blip from the map
                } else if(distance > DELETE_DISTANCE || !plane.isAlive) { //if the plane is in deleting distance or if it accidentally blew up
                    foreach(Ped pilot in pilots) { // mark all Peds as no longer needed so the game delets them
                        pilot.NoLongerNeeded();
                    }
                    plane.NoLongerNeeded();
                    plane.Delete(); // delete the plane blip from the map
                    pilots.Clear(); // clear our record of all Peds
                }
            }
        }

        // overloaded function for ConsoleCommandDelegate
        private void SpawnPlane(ParameterCollection Parameter) {
            SpawnPlane();
        }

        private void SpawnPlane() {
            Blip waypoint = Game.GetWaypoint();
            if(Exists(waypoint)) { // if a waypoint exists on the map
                // grab the waypoint's coordinates and set the plane's start and destination coordinates
                Vector3 startCoords = waypoint.Position.Around(PLANE_START_OFFSET).ToGround();
                destination = waypoint.Position.ToGround();
                destination.Z += DESTINATION_Z_OFFSET;
                startCoords.Z += PLANE_START_Z_OFFSET;

                // spawn the plane and attach its blip
                plane = World.CreateVehicle("ANNIHILATOR", startCoords);
                planeBlip = plane.AttachBlip();
                planeBlip.Friendly = true;
                planeBlip.Name = "Airstrike";

                // get the unit vector between the two points and set the plane's heading
                Vector3 startToDestination = destination - startCoords;
                plane.Heading = Helper.DirectionToHeading(startToDestination / startToDestination.Length());

                // add the pilot and passengers to the plane
                pilots.Add(plane.CreatePedOnSeat(VehicleSeat.Driver));
                pilots.Add(plane.CreatePedOnSeat(VehicleSeat.LeftRear));
                pilots.Add(plane.CreatePedOnSeat(VehicleSeat.RightRear));

                // create a fly sequence for the plane to its destination then to a random point
                FlySequence = new TaskSequence();
                FlySequence.AddTask.DriveTo(destination, 1000.0f, false);
                FlySequence.AddTask.DriveTo(Vector3.RandomXYZ(), 1000.0f, false);

                // assign the fly sequence to the pilot only
                ((Ped) pilots[0]).Task.PerformSequence(FlySequence);

                explosionDone = false;

                Game.DisplayText("Cheat activated: Airstrike is on its way.");
            } else { // if no waypoint exists on the map
                Game.DisplayText("Cheat not activated: Add a waypoint on the map to activate airstrike.");
            }
        }
    }
}